#!/usr/bin/env python
import datetime
import json
import re

from ansible.module_utils.urls import open_url
from ansible.plugins.lookup import LookupBase
from bs4 import BeautifulSoup

MINICONDA = "miniconda"
ANACONDA = "anaconda"

BASE_URL = {
    MINICONDA: "https://repo.continuum.io/miniconda/",
    ANACONDA: "https://repo.anaconda.com/archive/",
}
PATTERN_FILENAME = {
    MINICONDA: re.compile(
        r"Miniconda(?P<python>\d+)-(?P<version>[0-9]+.[0-9]+.[0-9]+(.[0-9]+)?|latest)-(?P<os>\w+)-(?P<arch>\w+).\w+"),
    ANACONDA: re.compile(
        r"Anaconda(?P<python>\d+)-(?P<version>[0-9]+.[0-9]+(.[0-9]+)?|latest)-(?P<os>\w+)-(?P<arch>\w+).\w+"),
}
RELEASE_MAX_AGE_DAYS = 365


def fetch_releases(distribution=MINICONDA, architecture="x86_64", python_version="3",
                   release_max_age_days=RELEASE_MAX_AGE_DAYS):
    base_url = BASE_URL[distribution]
    request = open_url(url=base_url, method="GET")
    response = request.read()

    data = {}
    soup = BeautifulSoup(response, "html.parser")
    table = soup.find('table')
    rows = table.find_all('tr')
    header = [element.text for element in rows[0].find_all('th')]
    now = datetime.datetime.now()

    for row in rows[1:]:
        cols = row.find_all('td')
        cols = [ele.text.strip() for ele in cols]
        release = {key: value for key, value in zip(header, cols)}
        result = PATTERN_FILENAME[distribution].match(release["Filename"])
        if result:
            timestamp = datetime.datetime.strptime(release["Last Modified"], "%Y-%m-%d %H:%M:%S")
            if (now - timestamp).days > release_max_age_days:
                continue
            info = {
                "python": result.group("python"),
                "version": result.group("version"),
                "os": result.group("os"),
                "arch": result.group("arch"),
                "download": "{base}{filename}".format(base=base_url, filename=release["Filename"])
            }
            release.update(info)
            if release["os"] == "Linux" and release['arch'] == architecture and release["python"] == str(python_version):
                data[release["version"]] = release

    if len(data) == 0:
        return data  # nothing found

    # find latest version number
    if distribution == MINICONDA:
        latest_item = [item for item in data.values() if item["version"] == "latest"]
        latest_version = [item for item in data.values() if
                          item["MD5"] == latest_item[0]["MD5"] and item["version"] != "latest"]
        latest_version_str = latest_version[0]["version"]
        # update latest version number in map
        data["latest"]["version"] = latest_version_str
    elif distribution == ANACONDA:
        sorted_versions = sorted(data.keys(), reverse=True)
        latest_version = sorted_versions[0]
        data["latest"] = data[latest_version]

    return data


def parse_plugin_parameter(terms, position, default, allowed_terms):
    result = default
    if len(terms) > position and terms[position] in allowed_terms:
        result = terms[position]
    return result


class LookupModule(LookupBase):
    def run(self, terms, variables=None, **kwargs):
        conda_distribution = parse_plugin_parameter(terms, position=0, default=MINICONDA,
                                                    allowed_terms=[MINICONDA, ANACONDA])
        python_version = parse_plugin_parameter(terms, position=1, default="3",
                                                allowed_terms=["2", "3", 2, 3])

        release_data = fetch_releases(distribution=conda_distribution, python_version=python_version)
        # if len(release_data) == 0:
        #    raise AnsibleError("Failed to lookup conda releases for %s" % conda_distribution)
        return [json.dumps(release_data)]


if __name__ == '__main__':
    print(json.dumps({"Miniconda for python 3": fetch_releases(distribution=MINICONDA, python_version="3")}, indent=4))
    print(json.dumps({"Anaconda for python 2": fetch_releases(distribution=ANACONDA, python_version="2")}, indent=4))
