#!/usr/bin/env python
import datetime
import json
import re

from ansible.module_utils.urls import open_url
from ansible.plugins.lookup import LookupBase
from bs4 import BeautifulSoup

URL_ANACONDA = "https://repo.anaconda.com/archive/"
URL_MINICONDA = "https://repo.continuum.io/miniconda/"
PATTERN_PYTHON = re.compile(
    r"Miniconda(?P<python>\d+)-(?P<version>[0-9]+.[0-9]+.[0-9]+|latest)-(?P<os>\w+)-(?P<arch>\w+).\w+")
RELEASE_MAX_AGE_DAYS = 365


def fetch_miniconda_releases(architecture="x86_64", python_version="3", release_max_age_days=RELEASE_MAX_AGE_DAYS):
    request = open_url(url=URL_MINICONDA, method="GET")
    response = request.read()

    data = {}
    soup = BeautifulSoup(response, "html.parser")
    table = soup.find('table')
    rows = table.find_all('tr')
    header = [element.text for element in rows[0].find_all('th')]
    now = datetime.datetime.now()

    for row in rows[1:]:
        cols = row.find_all('td')
        cols = [ele.text.strip() for ele in cols]
        release = {key: value for key, value in zip(header, cols)}
        result = PATTERN_PYTHON.match(release["Filename"])
        if result:
            info = {
                "python": result.group("python"),
                "version": result.group("version"),
                "os": result.group("os"),
                "arch": result.group("arch"),
                "download": "{base}{filename}".format(base=URL_MINICONDA, filename=release["Filename"])
            }
            release.update(info)
            timestamp = datetime.datetime.strptime(release["Last Modified"], "%Y-%m-%d %H:%M:%S")
            if (now - timestamp).days > release_max_age_days:
                continue
            if release["os"] == "Linux" and release['arch'] == architecture and release["python"] == python_version:
                data[release["version"]] = release

    # find actual latest version number
    latest_item = [item for item in data.values() if item["version"] == "latest"]
    latest_version = [item for item in data.values() if
                      item["MD5"] == latest_item[0]["MD5"] and item["version"] != "latest"]
    latest_version_str = latest_version[0]["version"]
    # update latest version number in map
    data["latest"]["version"] = latest_version_str

    return data


class LookupModule(LookupBase):
    def run(self, terms, variables=None, **kwargs):
        return [json.dumps(fetch_miniconda_releases())]


if __name__ == '__main__':
    print(json.dumps(fetch_miniconda_releases(), indent=4))
